import http from 'k6/http';
import { check, sleep } from 'k6';
import { Trend, Counter } from 'k6/metrics';

// Custom metrics (will appear in InfluxDB)
export let ResponseTime = new Trend('custom_response_time');
export let FailedRequests = new Counter('custom_failed_requests');

export let options = {
  stages: [
    { duration: '30s', target: 10 },    // ramp-up users
    { duration: '1m', target: 10 },     // stay at 10 users
    { duration: '30s', target: 0 },     // ramp-down
  ],

  // Performance thresholds
  thresholds: {
    http_req_duration: ['p(95)<500'], // 95% of requests < 500ms
    http_req_failed: ['rate<0.02'],   // <2% request errors
  },
};

// Base URL you want to test
const BASE_URL = 'https://web.app';  // change to your service

export default function () {
  let res = http.get(`${BASE_URL}/`);

  // Store custom metrics
  ResponseTime.add(res.timings.duration);

  // Validation
  const success = check(res, {
    'status is 200': (r) => r.status === 200,
  });

  if (!success) {
    FailedRequests.add(1);
  }

  sleep(1);
}

// Optional: summary printed in terminal after the test ends
export function handleSummary(data) {
  return {
    stdout: `
==========================================
     K6 LOAD TEST SUMMARY  
==========================================
Duration: ${data.state.testRunDuration}
Total Requests: ${data.metrics.http_reqs.values.count}
95th Percentile (ms): ${data.metrics.http_req_duration.values['p(95)']}
Failures: ${data.metrics.http_req_failed.values.rate * 100} %
==========================================
`,
  };
}
